# IBrandBiz – Outline & Sections (Core Build)

Here’s the expanded spec + scaffold with more structured templates and section support.

---

## Default Plan Outline (Must-Have Sections)
1. **Executive Summary**
2. **Company Overview**
   - Mission & Vision
   - Legal Structure
   - Founding Team
3. **Products & Services**
   - Product/Service Description
   - Value Proposition
   - Roadmap
4. **Market Analysis**
   - Industry Overview
   - Target Market (TAM/SAM/SOM)
   - Competitive Landscape
   - SWOT Analysis (Template)
5. **Marketing & Sales**
   - Customer Personas (Template)
   - Go-to-Market Strategy
   - Pricing Model (Template)
   - Sales Channels
6. **Operations Plan**
   - Key Activities
   - Resources & Suppliers
   - Technology Stack
7. **Organization & Management**
   - Org Chart
   - Roles & Responsibilities
   - Advisors
8. **Financial Plan**
   - Revenue Model
   - Cost Structure
   - 3-Year Projections (tables)
   - Break-even Analysis
   - Funding Requirements

---

## Templates (Fill-in-the-Blank)

### SWOT Analysis
- **Strengths:** [List internal advantages]
- **Weaknesses:** [List internal challenges]
- **Opportunities:** [List external trends/opportunities]
- **Threats:** [List external risks]

### Customer Persona
- **Name:** [Persona label]
- **Demographics:** Age, gender, location
- **Goals:** [Primary objectives]
- **Frustrations:** [Pain points]
- **Buying Triggers:** [Motivators]

### TAM / SAM / SOM
- **TAM (Total Addressable Market):** [Global demand]
- **SAM (Serviceable Available Market):** [Reachable market]
- **SOM (Serviceable Obtainable Market):** [Market share goal]

### Pricing Table
| Tier | Features | Price |
|------|----------|-------|
| Free | Basic access | $0 |
| Pro | Premium features | $49/mo |
| Enterprise | Custom features & support | Contact us |

### Porter’s Five Forces
1. Threat of New Entrants
2. Bargaining Power of Suppliers
3. Bargaining Power of Buyers
4. Threat of Substitutes
5. Industry Rivalry

---

## Drag-and-Drop Editor Features
- Reorder sections and sub-sections (tree-based)
- Add new section or template block
- Duplicate/delete sections
- Inline rename

---

## AI Prompt Hooks
- **Outline Generator:**
  > “Generate a structured business plan outline for [industry], with 6–10 sections and optional subsections.”

- **Section Drafting:**
  > “Write a [tone: Professional/Friendly/Bold/Minimal] draft for the section [name], using inputs: [user notes]. Return markdown paragraphs only.”

---

## Next Dev Steps
- Persist outline (localStorage → API)
- UI rendering with shadcn (Cards, Tree View)
- Live preview of templates → markdown → rendered prose
- Wire export (PDF/DOCX/Google Docs)

---

/* =========================
   V2 ADDITIONS – nested subsections, persistence, more templates
   ========================= */

/* =========================
   File: src/state/usePlanStore.v2.ts
   ========================= */
import { create as createZ } from "zustand";
import { nanoid as nid } from "nanoid";
import type { PlanState, PlanSection, SectionKind, SectionId } from "../types/plan";

const V2_STORAGE_KEY = "ibrandbiz.plan.v2";

const V2_DEFAULT_SECTIONS: PlanSection[] = [
  { id: nid(), kind: "executive-summary", title: "Executive Summary", content: "" },
  { id: nid(), kind: "company-overview", title: "Company Overview", content: "" },
  { id: nid(), kind: "products-services", title: "Products & Services", content: "" },
  { id: nid(), kind: "market-analysis", title: "Market Analysis", content: "" },
  { id: nid(), kind: "marketing-sales", title: "Marketing & Sales", content: "" },
  { id: nid(), kind: "operations", title: "Operations Plan", content: "" },
  { id: nid(), kind: "org-management", title: "Organization & Management", content: "" },
  { id: nid(), kind: "financials", title: "Financial Plan", content: "" },
];

function v2Load(): PlanState {
  try {
    const raw = localStorage.getItem(V2_STORAGE_KEY);
    if (raw) return JSON.parse(raw) as PlanState;
  } catch {}
  return { planId: nid(), title: "Untitled Business Plan", sections: V2_DEFAULT_SECTIONS };
}

type V2Store = PlanState & {
  addSection(kind?: SectionKind, title?: string): SectionId;
  addChildSection(parentId: SectionId, title?: string): SectionId | null;
  removeSection(id: SectionId): void;
  removeChildSection(parentId: SectionId, childId: SectionId): void;
  reorderSections(idsInOrder: SectionId[]): void;
  reorderChildSections(parentId: SectionId, idsInOrder: SectionId[]): void;
  updateSection(id: SectionId, patch: Partial<PlanSection>): void;
  updateChildSection(parentId: SectionId, childId: SectionId, patch: Partial<PlanSection>): void;
  duplicateSection(id: SectionId): SectionId | null;
  resetToDefault(): void;
};

export const usePlanStoreV2 = createZ<V2Store>((set, get) => ({
  ...v2Load(),
  addSection: (kind = "custom", title = "New Section") => {
    const sec: PlanSection = { id: nid(), kind, title, content: "", children: [] };
    set((s) => ({ sections: [...s.sections, sec] }));
    v2Persist();
    return sec.id;
  },
  addChildSection: (parentId, title = "New Subsection") => {
    set((s) => ({
      sections: s.sections.map((sec) =>
        sec.id === parentId
          ? { ...sec, children: [...(sec.children || []), { id: nid(), kind: "custom", title, content: "" }] }
          : sec
      ),
    }));
    v2Persist();
    const p = get().sections.find((x) => x.id === parentId);
    const last = p?.children?.[p.children.length - 1];
    return last?.id || null;
  },
  removeSection: (id) => { set((s) => ({ sections: s.sections.filter((x) => x.id !== id) })); v2Persist(); },
  removeChildSection: (parentId, childId) => {
    set((s) => ({
      sections: s.sections.map((sec) => sec.id === parentId ? { ...sec, children: (sec.children||[]).filter((c)=>c.id!==childId)}: sec)
    }));
    v2Persist();
  },
  reorderSections: (ids) => { set((s) => ({ sections: ids.map((id) => s.sections.find((x)=>x.id===id)!).filter(Boolean) })); v2Persist(); },
  reorderChildSections: (parentId, ids) => {
    set((s) => ({
      sections: s.sections.map((sec) => {
        if (sec.id !== parentId) return sec;
        const current = sec.children || [];
        const out = ids.map((id) => current.find((c)=>c.id===id)!).filter(Boolean);
        return { ...sec, children: out };
      })
    }));
    v2Persist();
  },
  updateSection: (id, patch) => { set((s)=>({ sections: s.sections.map((sec)=>sec.id===id?{...sec, ...patch}:sec) })); v2Persist(); },
  updateChildSection: (parentId, childId, patch) => {
    set((s)=>({ sections: s.sections.map((sec)=> sec.id!==parentId? sec : { ...sec, children: (sec.children||[]).map((c)=> c.id===childId?{...c, ...patch}:c) }) }));
    v2Persist();
  },
  duplicateSection: (id) => {
    const src = get().sections.find((x)=>x.id===id); if (!src) return null;
    const dupe: PlanSection = { ...src, id: nid(), title: `${src.title} (Copy)` };
    set((s)=>({ sections: [...s.sections, dupe] })); v2Persist(); return dupe.id;
  },
  resetToDefault: () => {
    const state = { planId: nid(), title: "Untitled Business Plan", sections: V2_DEFAULT_SECTIONS };
    set(state); try { localStorage.setItem(V2_STORAGE_KEY, JSON.stringify(state)); } catch {}
  },
}));

function v2Persist(){
  try {
    const { planId, title, sections } = usePlanStoreV2.getState();
    localStorage.setItem(V2_STORAGE_KEY, JSON.stringify({ planId, title, sections }));
  } catch {}
}

/* =========================
   File: src/templates/structured.v2.ts
   ========================= */
import type { StructuredTemplate } from "../types/plan";

export const V2_TEMPLATES: StructuredTemplate[] = [
  {
    key: "porter5",
    name: "Porter’s Five Forces",
    description: "Analyze industry competitiveness.",
    fields: [
      { id: "threat_new", label: "Threat of New Entrants", type: "textarea" },
      { id: "bargain_sup", label: "Bargaining Power of Suppliers", type: "textarea" },
      { id: "bargain_buy", label: "Bargaining Power of Buyers", type: "textarea" },
      { id: "threat_sub", label: "Threat of Substitutes", type: "textarea" },
      { id: "rivalry", label: "Industry Rivalry", type: "textarea" },
    ],
  },
  {
    key: "tam_sam_som",
    name: "TAM / SAM / SOM",
    description: "Market sizing overview.",
    fields: [
      { id: "tam", label: "TAM (Total Addressable Market)", type: "textarea", placeholder: "Size, assumptions, sources" },
      { id: "sam", label: "SAM (Serviceable Available Market)", type: "textarea" },
      { id: "som", label: "SOM (Serviceable Obtainable Market)", type: "textarea" },
      { id: "sources", label: "Sources / Citations", type: "textarea" },
    ],
  },
  {
    key: "pricing_table",
    name: "Pricing Table",
    description: "List tiers or SKUs with price & notes.",
    fields: [
      { id: "rows", label: "Rows (Name – Price – Notes)", type: "textarea", placeholder: "Basic – $19/mo – For starters
Pro – $49/mo – Most popular
Enterprise – $199/mo – Custom SLAs" },
    ],
  },
];

export function renderV2TemplateMarkdown(key: string, data: Record<string,string>): string {
  if (key === "porter5") return `### Porter’s Five Forces

**Threat of New Entrants**
${data.threat_new||"-"}

**Bargaining Power of Suppliers**
${data.bargain_sup||"-"}

**Bargaining Power of Buyers**
${data.bargain_buy||"-"}

**Threat of Substitutes**
${data.threat_sub||"-"}

**Industry Rivalry**
${data.rivalry||"-"}`;
  if (key === "tam_sam_som") return `### Market Size: TAM / SAM / SOM

**TAM**
${data.tam||"-"}

**SAM**
${data.sam||"-"}

**SOM**
${data.som||"-"}

**Sources**
${data.sources||"-"}`;
  if (key === "pricing_table") {
    const rows = (data.rows||"").split(/
+/).filter(Boolean);
    const table = ["| Name | Price | Notes |", "|---|---:|---|"]
      .concat(rows.map((r)=>{ const [name="", price="", notes=""] = r.split(" – "); return `| ${name.trim()} | ${price.trim()} | ${notes.trim()} |`; }))
      .join("
");
    return `### Pricing

${table}`;
  }
  return "";
}

/* =========================
   File: src/components/OutlineEditor.v2.tsx
   ========================= */
import React from "react";
import { DndContext, closestCenter, PointerSensor, useSensor, useSensors } from "@dnd-kit/core";
import { arrayMove, SortableContext, verticalListSortingStrategy } from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";
import { useSortable } from "@dnd-kit/sortable";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";

function Row({ id, title, parentId }: { id: string; title: string; parentId?: string }) {
  const { attributes, listeners, setNodeRef, transform, transition, isDragging } = useSortable({ id });
  const update = usePlanStoreV2((s)=> s.updateSection);
  const updateChild = usePlanStoreV2((s)=> s.updateChildSection);
  const remove = usePlanStoreV2((s)=> s.removeSection);
  const removeChild = usePlanStoreV2((s)=> s.removeChildSection);
  const addChild = usePlanStoreV2((s)=> s.addChildSection);

  const style: React.CSSProperties = { transform: CSS.Transform.toString(transform), transition, opacity: isDragging?0.6:1 };
  const onRename = (val:string)=> parentId ? updateChild(parentId, id, { title: val }) : update(id, { title: val });
  const onDelete = ()=> parentId ? removeChild(parentId, id) : remove(id);

  return (
    <div ref={setNodeRef} style={style} className="flex items-center justify-between rounded-xl border p-2">
      <div className="flex items-center gap-3">
        <button {...attributes} {...listeners} className="cursor-grab">≡</button>
        <input className="border-none outline-none bg-transparent text-base" defaultValue={title} onBlur={(e)=> onRename(e.currentTarget.value)} />
      </div>
      <div className="flex gap-2">
        {!parentId && (<button onClick={()=> addChild(id)} className="px-2 py-1 rounded-lg border">Add Subsection</button>)}
        <button onClick={onDelete} className="px-2 py-1 rounded-lg border text-red-600">Delete</button>
      </div>
    </div>
  );
}

export function OutlineEditorV2(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const reorder = usePlanStoreV2((s)=> s.reorderSections);
  const reorderChild = usePlanStoreV2((s)=> s.reorderChildSections);
  const sensors = useSensors(useSensor(PointerSensor));

  function onDragEndTop(event:any){
    const { active, over } = event; if (!over || active.id===over.id) return;
    const ids = sections.map((s)=> s.id); const oldIndex = ids.indexOf(active.id); const newIndex = ids.indexOf(over.id);
    reorder(arrayMove(ids, oldIndex, newIndex));
  }
  return (
    <div className="space-y-3">
      <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={onDragEndTop}>
        <SortableContext items={sections.map((s)=> s.id)} strategy={verticalListSortingStrategy}>
          {sections.map((s)=> (
            <div key={s.id} className="rounded-2xl border p-3">
              <Row id={s.id} title={s.title} />

              {(s.children||[]).length>0 && (
                <ChildList parentId={s.id} />
              )}
            </div>
          ))}
        </SortableContext>
      </DndContext>
    </div>
  );

  function ChildList({ parentId }:{ parentId: string }){
    const parent = sections.find((x)=> x.id===parentId);
    const children = parent?.children || [];

    function onDragEndChild(event:any){
      const { active, over } = event; if (!over || active.id===over.id) return;
      const ids = children.map((c)=> c.id); const oldIndex = ids.indexOf(active.id); const newIndex = ids.indexOf(over.id);
      reorderChild(parentId, arrayMove(ids, oldIndex, newIndex));
    }
    return (
      <div className="mt-3 ml-7">
        <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={onDragEndChild}>
          <SortableContext items={children.map((c)=> c.id)} strategy={verticalListSortingStrategy}>
            <div className="space-y-2">
              {children.map((c)=> (<Row key={c.id} id={c.id} title={c.title} parentId={parentId} />))}
            </div>
          </SortableContext>
        </DndContext>
      </div>
    );
  }
}

/* =========================
   File: src/components/StructuredForm.v2.tsx
   ========================= */
import React, { useState } from "react";
import { V2_TEMPLATES, renderV2TemplateMarkdown } from "../templates/structured.v2";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";

export function StructuredFormV2({ targetSectionId }:{ targetSectionId: string }){
  const [selectedKey, setSelectedKey] = useState<string>(V2_TEMPLATES[0].key);
  const [data, setData] = useState<Record<string,string>>({});
  const update = usePlanStoreV2((s)=> s.updateSection);

  const tpl = V2_TEMPLATES.find((t)=> t.key===selectedKey)!;
  function insert(){
    const md = renderV2TemplateMarkdown(tpl.key, data);
    update(targetSectionId, (prev=> ({ content: `${(prev as any)?.content? (prev as any).content + "

" : ""}${md}` })) as any);
  }

  return (
    <div className="space-y-3">
      <select value={selectedKey} onChange={(e)=> setSelectedKey(e.target.value)} className="border rounded-xl p-2">
        {V2_TEMPLATES.map((t)=> (<option key={t.key} value={t.key}>{t.name}</option>))}
      </select>
      <div className="grid gap-2">
        {tpl.fields.map((f)=> (
          <label key={f.id} className="grid gap-1">
            <span className="text-sm text-gray-600">{f.label}</span>
            {f.type === "textarea" ? (
              <textarea rows={3} className="rounded-xl border p-2" placeholder={f.placeholder} onChange={(e)=> setData((d)=> ({...d, [f.id]: e.target.value}))} />
            ) : (
              <input className="rounded-xl border p-2" placeholder={f.placeholder} onChange={(e)=> setData((d)=> ({...d, [f.id]: e.target.value}))} />
            )}
          </label>
        ))}
      </div>
      <button onClick={insert} className="rounded-2xl border px-3 py-2">Insert into Section</button>
    </div>
  );
}

/* =========================
   File: src/App.v2.tsx (integration sketch)
   ========================= */
import React from "react";
import { OutlineEditorV2 } from "./components/OutlineEditor.v2";
import { usePlanStoreV2 } from "./state/usePlanStore.v2";
import { StructuredFormV2 } from "./components/StructuredForm.v2";

export default function AppV2(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const update = usePlanStoreV2((s)=> s.updateSection);
  const first = sections[0];

  return (
    <div className="grid md:grid-cols-2 gap-6 p-6">
      <div>
        <h2 className="text-xl font-semibold mb-3">Outline (V2)</h2>
        <OutlineEditorV2 />
      </div>
      <div>
        <h2 className="text-xl font-semibold mb-3">Structured Templates (V2)</h2>
        {first && <StructuredFormV2 targetSectionId={first.id} />}
        {first && (
          <div className="mt-6">
            <h3 className="font-medium mb-2">Live Preview (first section content)</h3>
            <textarea className="w-full h-48 rounded-xl border p-3" value={first.content} onChange={(e)=> update(first.id, { content: e.target.value })} />
          </div>
        )}
      </div>
    </div>
  );
}


/* =========================
   FINANCIALS – Deterministic Inputs + 12‑Month P&L
   ========================= */

/* =========================
   File: src/types/finance.ts
   ========================= */
export type Currency = "USD" | "EUR" | "GBP" | "CAD" | "AUD";

export interface FinanceInputs {
  currency: Currency;
  startMonth: string; // YYYY-MM (e.g., "2025-10")
  pricePerUnit: number; // average selling price
  unitsPerMonth: number[]; // length 12, units sold per month
  cogsPercent: number; // 0..100
  fixedMonthlyExpenses: number; // rent, tools, etc.
  variableExpensePerUnit?: number; // optional extra variable cost
  startingCash?: number; // for runway calc
}

export interface MonthlyRow {
  month: string; // e.g., "Oct 2025"
  revenue: number;
  cogs: number;
  grossProfit: number;
  opex: number; // fixedMonthlyExpenses
  profit: number; // grossProfit - opex
  cumulativeProfit: number;
}

export interface FinanceOutputs {
  rows: MonthlyRow[]; // 12 months
  totalRevenue: number;
  totalProfit: number;
  breakevenMonthIndex: number | null; // 0..11 or null
  runwayMonths: number | null; // if startingCash provided
}

/* =========================
   File: src/lib/financeCalc.ts
   ========================= */
import type { FinanceInputs, FinanceOutputs, MonthlyRow } from "../types/finance";

const MONTH_NAMES = ["Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"];

function formatMonthLabel(ym: string, i: number) {
  // ym = YYYY-MM
  const [y0, m0] = ym.split("-").map(Number);
  const base = new Date(y0, (m0 - 1) + i, 1);
  return `${MONTH_NAMES[base.getMonth()]} ${base.getFullYear()}`;
}

export function computePAndL(input: FinanceInputs): FinanceOutputs {
  const rows: MonthlyRow[] = [];
  let cumulative = 0;
  const price = Number(input.pricePerUnit) || 0;
  const cogsPct = Math.min(Math.max(input.cogsPercent, 0), 100) / 100;
  const varPerUnit = Number(input.variableExpensePerUnit || 0);
  const opex = Number(input.fixedMonthlyExpenses) || 0;

  for (let i = 0; i < 12; i++) {
    const units = Number(input.unitsPerMonth[i] || 0);
    const revenue = units * price;
    const cogs = revenue * cogsPct + units * varPerUnit;
    const grossProfit = revenue - cogs;
    const profit = grossProfit - opex;
    cumulative += profit;

    rows.push({
      month: formatMonthLabel(input.startMonth, i),
      revenue, cogs, grossProfit, opex, profit, cumulativeProfit: cumulative,
    });
  }

  const totalRevenue = rows.reduce((s, r) => s + r.revenue, 0);
  const totalProfit = rows.reduce((s, r) => s + r.profit, 0);

  // breakeven = first month where cumulativeProfit >= 0
  let breakevenMonthIndex: number | null = null;
  for (let i = 0; i < rows.length; i++) {
    if (rows[i].cumulativeProfit >= 0) { breakevenMonthIndex = i; break; }
  }

  let runwayMonths: number | null = null;
  if (input.startingCash != null) {
    let cash = input.startingCash;
    let m = 0;
    while (m < 60) { // cap at 5 years
      const idx = Math.min(m, rows.length - 1);
      cash += rows[idx].profit;
      if (cash <= 0) { runwayMonths = m + 1; break; }
      m++;
    }
    if (runwayMonths == null) runwayMonths = 60; // survival beyond horizon
  }

  return { rows, totalRevenue, totalProfit, breakevenMonthIndex, runwayMonths };
}

/* =========================
   File: src/state/useFinanceStore.ts
   ========================= */
import { create } from "zustand";
import type { FinanceInputs, FinanceOutputs } from "../types/finance";
import { computePAndL } from "../lib/financeCalc";

const KEY = "ibrandbiz.finance.v1";

export type FinanceStore = {
  inputs: FinanceInputs;
  outputs: FinanceOutputs | null;
  setInputs: (patch: Partial<FinanceInputs>) => void;
  recompute: () => void;
  reset: () => void;
};

const DEFAULT_INPUTS: FinanceInputs = {
  currency: "USD",
  startMonth: "2025-10",
  pricePerUnit: 49,
  unitsPerMonth: [10,12,15,18,22,26,30,35,40,46,52,60],
  cogsPercent: 30,
  fixedMonthlyExpenses: 3000,
  variableExpensePerUnit: 0,
  startingCash: 5000,
};

function load(): FinanceInputs {
  try { const raw = localStorage.getItem(KEY); if (raw) return JSON.parse(raw); } catch {}
  return DEFAULT_INPUTS;
}

export const useFinanceStore = create<FinanceStore>((set, get) => ({
  inputs: load(),
  outputs: null,
  setInputs: (patch) => {
    set((s) => ({ inputs: { ...s.inputs, ...patch } }));
    persist(get().inputs);
  },
  recompute: () => {
    const out = computePAndL(get().inputs);
    set({ outputs: out });
  },
  reset: () => {
    set({ inputs: DEFAULT_INPUTS, outputs: null });
    persist(DEFAULT_INPUTS);
  },
}));

function persist(i: FinanceInputs) { try { localStorage.setItem(KEY, JSON.stringify(i)); } catch {} }

/* =========================
   File: src/components/FinanceInputs.tsx
   ========================= */
import React, { useEffect } from "react";
import { useFinanceStore } from "../state/useFinanceStore";

export function FinanceInputsPanel(){
  const { inputs, setInputs, recompute, outputs } = useFinanceStore();
  useEffect(()=>{ recompute(); }, []);

  const updateUnits = (idx:number, val:number) => {
    const arr = [...inputs.unitsPerMonth];
    arr[idx] = val;
    setInputs({ unitsPerMonth: arr });
    recompute();
  };

  return (
    <div className="space-y-4">
      <div className="grid grid-cols-2 gap-3">
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Start Month (YYYY-MM)</span>
          <input className="border rounded-xl p-2" value={inputs.startMonth} onChange={(e)=>{setInputs({ startMonth: e.target.value }); recompute();}} />
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Currency</span>
          <select className="border rounded-xl p-2" value={inputs.currency} onChange={(e)=>{setInputs({ currency: e.target.value as any }); recompute();}}>
            {(["USD","EUR","GBP","CAD","AUD"] as const).map(c=> <option key={c} value={c}>{c}</option>)}
          </select>
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Price per Unit</span>
          <input type="number" className="border rounded-xl p-2" value={inputs.pricePerUnit} onChange={(e)=>{setInputs({ pricePerUnit: Number(e.target.value) }); recompute();}} />
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">COGS %</span>
          <input type="number" className="border rounded-xl p-2" value={inputs.cogsPercent} onChange={(e)=>{setInputs({ cogsPercent: Number(e.target.value) }); recompute();}} />
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Fixed Monthly Expenses</span>
          <input type="number" className="border rounded-xl p-2" value={inputs.fixedMonthlyExpenses} onChange={(e)=>{setInputs({ fixedMonthlyExpenses: Number(e.target.value) }); recompute();}} />
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Variable Expense per Unit (optional)</span>
          <input type="number" className="border rounded-xl p-2" value={inputs.variableExpensePerUnit||0} onChange={(e)=>{setInputs({ variableExpensePerUnit: Number(e.target.value) }); recompute();}} />
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Starting Cash (optional)</span>
          <input type="number" className="border rounded-xl p-2" value={inputs.startingCash||0} onChange={(e)=>{setInputs({ startingCash: Number(e.target.value) }); recompute();}} />
        </label>
      </div>

      <div className="mt-2">
        <span className="text-sm text-gray-600">Units per Month (12)
        </span>
        <div className="grid grid-cols-6 gap-2 mt-2">
          {Array.from({length:12}).map((_,i)=> (
            <input key={i} type="number" className="border rounded-lg p-2" value={inputs.unitsPerMonth[i]||0} onChange={(e)=> updateUnits(i, Number(e.target.value))} />
          ))}
        </div>
      </div>

      {outputs && (
        <div className="mt-4 grid gap-2">
          <div className="rounded-xl border p-3 text-sm">
            <div>Total Revenue: <strong>{inputs.currency} {outputs.totalRevenue.toLocaleString()}</strong></div>
            <div>Total Profit: <strong>{inputs.currency} {outputs.totalProfit.toLocaleString()}</strong></div>
            <div>Breakeven: <strong>{outputs.breakevenMonthIndex!=null ? outputs.rows[outputs.breakevenMonthIndex].month : "Not in 12 months"}</strong></div>
            {inputs.startingCash!=null && <div>Runway (est.): <strong>{outputs.runwayMonths} mo</strong></div>}
          </div>

          <div className="overflow-x-auto">
            <table className="w-full border-collapse border border-border">
              <thead className="bg-muted">
                <tr>
                  <th className="border p-2 text-left">Month</th>
                  <th className="border p-2 text-left">Revenue</th>
                  <th className="border p-2 text-left">COGS</th>
                  <th className="border p-2 text-left">Gross Profit</th>
                  <th className="border p-2 text-left">Opex</th>
                  <th className="border p-2 text-left">Profit</th>
                  <th className="border p-2 text-left">Cum. Profit</th>
                </tr>
              </thead>
              <tbody>
                {outputs.rows.map((r,idx)=> (
                  <tr key={idx}>
                    <td className="border p-2">{r.month}</td>
                    <td className="border p-2">{inputs.currency} {r.revenue.toLocaleString()}</td>
                    <td className="border p-2">{inputs.currency} {r.cogs.toLocaleString()}</td>
                    <td className="border p-2">{inputs.currency} {r.grossProfit.toLocaleString()}</td>
                    <td className="border p-2">{inputs.currency} {r.opex.toLocaleString()}</td>
                    <td className="border p-2">{inputs.currency} {r.profit.toLocaleString()}</td>
                    <td className="border p-2">{inputs.currency} {r.cumulativeProfit.toLocaleString()}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}
    </div>
  );
}

/* =========================
   File: src/components/FinanceSidebar.tsx
   ========================= */
import React from "react";
import { FinanceInputsPanel } from "./FinanceInputs";

export function FinanceSidebar(){
  return (
    <aside className="space-y-4">
      <h3 className="text-lg font-semibold">Financials</h3>
      <p className="text-sm text-muted-foreground">Enter a few assumptions to auto-generate your 12‑month P&L.</p>
      <FinanceInputsPanel />
    </aside>
  );
}

/* =========================
   AI WRITING ASSISTANT – Inline actions per section
   ========================= */

/* =========================
   File: src/types/ai.ts
   ========================= */
export type ToneOption = "Formal" | "Friendly" | "Persuasive" | "Inspirational" | "Technical";
export type AiAction = "generate" | "rephrase" | "expand" | "summarize";

export interface AiJob {
  action: AiAction;
  tone?: ToneOption;
  sectionTitle: string;
  companyName?: string;
  context?: string; // existing text or plan brief
}

/* =========================
   File: src/services/ai/aiClient.ts
   ========================= */
import type { AiJob } from "../../types/ai";

// Replace with your actual API route that calls GPT‑5 / GPT‑5‑mini
export async function callAi(job: AiJob): Promise<string> {
  const res = await fetch("/api/ai/section", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(job),
  });
  if (!res.ok) throw new Error("AI request failed");
  const data = await res.json();
  return data.text as string;
}

/* =========================
   File: src/components/AIButton.tsx
   ========================= */
import React, { useState } from "react";
import type { AiAction, ToneOption } from "../types/ai";
import { callAi } from "../services/ai/aiClient";

const TONES: ToneOption[] = ["Formal","Friendly","Persuasive","Inspirational","Technical"];

export function AIButton({
  sectionTitle,
  companyName,
  getContext,
  onApply,
}: {
  sectionTitle: string;
  companyName?: string;
  getContext: () => string; // current text
  onApply: (text: string) => void;
}){
  const [open, setOpen] = useState(false);
  const [loading, setLoading] = useState(false);
  const [tone, setTone] = useState<ToneOption>("Formal");
  const [action, setAction] = useState<AiAction>("generate");
  const [error, setError] = useState<string | null>(null);

  async function run(){
    setError(null); setLoading(true);
    try {
      const text = await callAi({ action, tone, sectionTitle, companyName, context: getContext() });
      onApply(text);
      setOpen(false);
    } catch(e:any){ setError(e?.message || "AI failed"); }
    finally { setLoading(false); }
  }

  return (
    <div className="relative inline-block">
      <button className="rounded-lg border px-2 py-1 text-sm" onClick={()=> setOpen((v)=> !v)}>Write with AI</button>
      {open && (
        <div className="absolute z-20 mt-2 w-64 rounded-xl border bg-white p-3 shadow-lg">
          <div className="grid gap-2">
            <label className="grid gap-1">
              <span className="text-xs text-gray-600">Action</span>
              <select className="border rounded-md p-1" value={action} onChange={(e)=> setAction(e.target.value as any)}>
                <option value="generate">Generate</option>
                <option value="rephrase">Rephrase</option>
                <option value="expand">Expand</option>
                <option value="summarize">Summarize</option>
              </select>
            </label>
            <label className="grid gap-1">
              <span className="text-xs text-gray-600">Tone</span>
              <select className="border rounded-md p-1" value={tone} onChange={(e)=> setTone(e.target.value as any)}>
                {TONES.map(t=> <option key={t} value={t}>{t}</option>)}
              </select>
            </label>
            {error && <div className="text-xs text-red-600">{error}</div>}
            <button className="rounded-md border px-2 py-1 text-sm" onClick={run} disabled={loading}>{loading? "Thinking…" : "Apply"}</button>
          </div>
        </div>
      )}
    </div>
  );
}

/* =========================
   File: src/components/SectionEditor.tsx
   ========================= */
import React from "react";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";
import { AIButton } from "./AIButton";

export function SectionEditor({ sectionId }:{ sectionId: string }){
  const section = usePlanStoreV2((s)=> s.sections.find(x=> x.id===sectionId));
  const update = usePlanStoreV2((s)=> s.updateSection);
  if (!section) return null;

  return (
    <div className="space-y-2">
      <div className="flex items-center justify-between">
        <h4 className="font-semibold">{section.title}</h4>
        <AIButton
          sectionTitle={section.title}
          companyName={"IBrandBiz"}
          getContext={() => section.content}
          onApply={(text) => update(sectionId, { content: text })}
        />
      </div>
      <textarea
        className="w-full h-48 rounded-xl border p-3"
        value={section.content}
        onChange={(e)=> update(sectionId, { content: e.target.value })}
        placeholder={`Write your ${section.title} here…`}
      />
    </div>
  );
}

/* =========================
   File: src/pages/Builder.tsx (integration sketch)
   ========================= */
import React from "react";
import { OutlineEditorV2 } from "../components/OutlineEditor.v2";
import { usePlanStoreV2 } from "../state/usePlanStore.v2";
import { SectionEditor } from "../components/SectionEditor";
import { FinanceSidebar } from "../components/FinanceSidebar";

export default function Builder(){
  const sections = usePlanStoreV2((s)=> s.sections);
  const first = sections[0];

  return (
    <div className="grid lg:grid-cols-[1.1fr_0.9fr] gap-8 p-6">
      <div className="space-y-6">
        <h2 className="text-xl font-semibold">Outline</h2>
        <OutlineEditorV2 />

        {first && (
          <div className="mt-6">
            <SectionEditor sectionId={first.id} />
          </div>
        )}
      </div>

      <div className="space-y-6">
        <FinanceSidebar />
      </div>
    </div>
  );
}

/* =========================
   File: src/pages/api/ai/section.ts (Next.js example)
   ========================= */
// import type { NextApiRequest, NextApiResponse } from "next";
// import type { AiJob } from "../../../types/ai";
// import OpenAI from "openai";
// const client = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
//
// export default async function handler(req: NextApiRequest, res: NextApiResponse){
//   if (req.method !== "POST") return res.status(405).end();
//   const job = req.body as AiJob;
//   const sys = "You are an expert business plan writer.";
//   const user = buildPrompt(job);
//   const r = await client.responses.create({ model: "gpt-5-mini", input: [{ role: "system", content: sys }, { role: "user", content: user }] });
//   // @ts-ignore
//   const text = r.output_text || r.output[0]?.content[0]?.text || "";
//   res.json({ text });
// }
//
// function buildPrompt(job: AiJob){
//   const base = `Section: ${job.sectionTitle}
Company: ${job.companyName||""}
Tone: ${job.tone||"Formal"}`;
//   if (job.action === "generate") return base + `
Write a concise, skimmable section (180–250 words).`;
//   if (job.action === "rephrase") return base + `
Rephrase the following to be clearer and more impactful, preserving meaning:

${job.context||""}`;
//   if (job.action === "expand") return base + `
Expand the following with 2–3 crisp paragraphs and a short bullet list:

${job.context||""}`;
//   if (job.action === "summarize") return base + `
Summarize the following into 3–5 bullets and a 2‑sentence close:

${job.context||""}`;
//   return base;
// }

